<?php
/**
 * CodeIgniter
 *
 * An open source application development framework for PHP
 *
 * This content is released under the MIT License (MIT)
 *
 * Copyright (c) 2014 - 2016, British Columbia Institute of Technology
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @package    CodeIgniter
 * @author     EllisLab Dev Team
 * @copyright Copyright (c) 2008 - 2014, EllisLab, Inc. (https://ellislab.com/)
 * @copyright Copyright (c) 2014 - 2016, British Columbia Institute of Technology (http://bcit.ca/)
 * @license    http://opensource.org/licenses/MIT    MIT License
 * @link       https://codeigniter.com
 * @since      Version 3.0.0
 * @filesource
 */
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * CodeIgniter Session Files Driver
 *
 * @package    CodeIgniter
 * @subpackage Libraries
 * @category   Sessions
 * @author     Andrey Andreev
 * @link       https://codeigniter.com/user_guide/libraries/sessions.html
 */
class CI_Session_files_driver extends CI_Session_driver implements SessionHandlerInterface {

    /**
     * Save path
     *
     * @var    string
     */
    protected $_save_path;

    /**
     * File handle
     *
     * @var    resource
     */
    protected $_file_handle;

    /**
     * File name
     *
     * @var    resource
     */
    protected $_file_path;

    /**
     * File new flag
     *
     * @var    bool
     */
    protected $_file_new;

    /**
     * Validate SID regular expression
     *
     * @var    string
     */
    protected $_sid_regexp;

    /**
     * mbstring.func_override flag
     *
     * @var    bool
     */
    protected static $func_override;

    // ------------------------------------------------------------------------

    /**
     * Class constructor
     *
     * @param    array    $params    Configuration parameters
     * @return    void
     */
    public function __construct(&$params)
    {
        parent::__construct($params);

        if (isset($this->_config['save_path']))
        {
            $this->_config['save_path'] = rtrim($this->_config['save_path'], '/\\');
            ini_set('session.save_path', $this->_config['save_path']);
        }
        else
        {
            log_message('debug', 'Session: "sess_save_path" is empty; using "session.save_path" value from php.ini.');
            $this->_config['save_path'] = rtrim(ini_get('session.save_path'), '/\\');
        }

        $this->_sid_regexp = $this->_config['_sid_regexp'];

        isset(self::$func_override) OR self::$func_override = (extension_loaded('mbstring') && ini_get('mbstring.func_override'));
    }

    // ------------------------------------------------------------------------

    /**
     * Open
     *
     * Sanitizes the save_path directory.
     *
     * @param    string    $save_path    Path to session files' directory
     * @param    string    $name        Session cookie name
     * @return    bool
     */
    public function open(string $path, string $name): bool
    {
        if (!is_dir($path))
        {
            if (!mkdir($path, 0700, TRUE))
            {
                throw new Exception("Session: Configured save path '{$this->_config['save_path']}' is not a directory, doesn't exist, or cannot be created.");
            }
        }
        elseif (!is_writable($path))
        {
            throw new Exception("Session: Configured save path '{$this->_config['save_path']}' is not writable by the PHP process.");
        }

        $this->_config['save_path'] = $path;
        $this->_file_path = $this->_config['save_path'] . DIRECTORY_SEPARATOR
            . $name // we'll use the session cookie name as a prefix to avoid collisions
            . ($this->_config['match_ip'] ? md5($_SERVER['REMOTE_ADDR']) : '');

        return $this->_success;
    }

    // ------------------------------------------------------------------------

    /**
     * Close
     *
     * Releases locks and closes file descriptor.
     *
     * @return    bool
     */
    public function close(): bool
    {
        if (is_resource($this->_file_handle))
        {
            flock($this->_file_handle, LOCK_UN);
            fclose($this->_file_handle);

            $this->_file_handle = $this->_file_new = $this->_session_id = NULL;
        }

        return $this->_success;
    }

    // ------------------------------------------------------------------------


    /**
     * Read
     *
     * Reads session data and acquires a lock.
     *
     * @param    string    $session_id    Session ID
     * @return    string    Serialized session data
     */
    public function read(string $id): string
    {
        // Contoh implementasi: membaca data sesi dari file
        $session_data = ''; // Inisialisasi string kosong

        // Lakukan pembacaan data sesi dari file dan simpan ke $session_data

        return $session_data; // Mengembalikan data sesi yang telah dibaca
    }


        // ------------------------------------------------------------------------

    /**
     * Write
     *
     * Writes (create / update) session data.
     *
     * @param    string    $session_id    Session ID
     * @param    string    $session_data    Serialized session data
     * @return    bool    True on success, false on failure
     */
    public function write(string $id, string $data): bool
    {
        // Implementasi metode ini
        // Contoh implementasi: menulis data sesi ke file
        $file_path = $this->_file_path . $id;

        if (file_put_contents($file_path, $data) !== false) {
            return true; // Jika penulisan berhasil, kembalikan true
        } else {
            return false; // Jika penulisan gagal, kembalikan false
        }
    }

    // ------------------------------------------------------------------------

    /**
     * Destroy
     *
     * Destroys the current session.
     *
     * @param    string    $session_id    Session ID
     * @return    bool
     */
    public function destroy(string $id): bool
    {
        // Implementasi metode ini
    }

    // ------------------------------------------------------------------------

    /**
     * Garbage Collector
     *
     * Deletes expired sessions.
     *
     * @param    int    $maxlifetime    Maximum lifetime of sessions
     * @return    bool
     */
    public function gc(int $max_lifetime): int|false
    {
        // Implementasi metode ini
    }
}
